#include "CameraCBuffer.hlsli"
#include "SamplerStates.hlsli"

// The rendertarget and hardware depth buffer
Texture2D RenderTargetTex : register(t0);
Texture2D<float2> DepthAndCoCRadiusTex : register(t1);

#include "Common.hlsli"

struct GSInput {
	float4 PositionClipS : SV_Position;
	float2 TexCoords : TEXCOORD0;
};

struct GSOutput
{
	float4 PositionClipS : SV_Position;
	float4 Color : Color;
	float2 TexCoords : TexCoord0;
	float SphericalAmount : SA;
};

[maxvertexcount(16)]
void main(point GSInput center[1], inout TriangleStream<GSOutput> triStream )
{
	float2 depthAndCoCRadius = DepthAndCoCRadiusTex.SampleLevel(LinearSampler, center[0].TexCoords, 0);
	float linearDepth = depthAndCoCRadius.r;

	// Convert CoC radius from film space (meters) to image space (full resolution pixels)
	float CoCRadiusFilmS = depthAndCoCRadius.g;
	float CoCDiameterPixelS = (2 * CoCRadiusFilmS) * ImageWidth / FilmWidth;
	
	// Only render if the CoC is wide enough, and clamp if the CoC is too big
	if(CoCDiameterPixelS > kCoCDiameterPixels) {
		CoCDiameterPixelS = kCoCDiameterPixels;
	}
	else if(CoCDiameterPixelS < kMinCoCPixels) {
		return;
	} 
	
	// Convert clamped diameter to clip space radius. Clip space values have a range of [-1, 1], 
	// so width for example would be (radius * 2 / imageWidth), or (diameter / imageWidth)
	float CoCXClipS = CoCDiameterPixelS / ImageWidth;
	float2 CoCRadiusClipS = float2(CoCXClipS, CoCXClipS * FilmWidth / FilmHeight);

	// The quad's alpha is proportional to the inverse of the radius or diameter squared.
	// Calculate this in image space (in terms of pixels).
	float alphaModifier = 1 / (CoCDiameterPixelS * CoCDiameterPixelS);

	float4 positionClipS = float4(center[0].PositionClipS.xy, 0, 1);
	float4 color = RenderTargetTex.SampleLevel(LinearSampler, center[0].TexCoords, 0);
	color.a *= alphaModifier;
	
	// Store the amount of SA to apply.
	// NOTE: A lens with undercorrected spherical aberration is associated with a smooth background blur and a harsh foreground blur.
	// Positive SA values will be used for smooth blur, and negative SA values for harsh blur.
	float sa = clamp(SphericalAberrationFactor * CoCDiameterPixelS / kCoCDiameterPixels, -1, 1);

	// Modify Z to indicate whether the point is in the foreground or background.
	if(linearDepth < FocalDistance) {
		// Foreground blur
		positionClipS.z = 0;

		// Negate the SA value to change the type of blur to be performed
		sa *= -1;
	} else {
		// Background blur
		positionClipS.z = 1;
	}

	GSOutput v = (GSOutput)0;
	v.Color = color;
	v.SphericalAmount = sa;

	// Expand the point into a quad
	//------------------------------
	v.PositionClipS = positionClipS;

	// bottom left
	v.PositionClipS.xy = positionClipS.xy - CoCRadiusClipS;
	v.TexCoords = float2(0, 1);
	triStream.Append(v);

	// top left
	v.PositionClipS.x = positionClipS.x - CoCRadiusClipS.x;
	v.PositionClipS.y = positionClipS.y + CoCRadiusClipS.y;
	v.TexCoords = float2(0, 0);
	triStream.Append(v);

	// bottom right
	v.PositionClipS.x = positionClipS.x + CoCRadiusClipS.x;
	v.PositionClipS.y = positionClipS.y - CoCRadiusClipS.y;
	v.TexCoords = float2(1, 1);
	triStream.Append(v);

	// top right
	v.PositionClipS.xy = positionClipS.xy + CoCRadiusClipS;
	v.TexCoords = float2(1, 0);
	triStream.Append(v);

	triStream.RestartStrip();
}
